//*************************************************************************************************
//
//	Description:
//		Basic generic quad shader
//		
//		
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Matt Hobbs
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		Matt             25/04/2007  Created
//
//	<TABLE>
//
//*************************************************************************************************

#if defined(_XBOX) && !defined(_TOOLS_COMPILATION_)
#include "D:\Render\Shaders\stddefs.fxh"
#else
#include "..\..\..\Render\Shaders\stddefs.fxh"
#endif

#if defined(_PS3_)
#define _MINFILTER	LinearMipMapLinear
#else
#define _MINFILTER	Linear
#define SET_NO_ANISOTROPY MaxAnisotropy = 1;
#endif

#define _SSAO_READY_
//-----------------------------------------------------------------------
//
// Transforms
//
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	bool appEdit = false;
>;

float4x4 worldviewproj : WorldViewProjection
<
	bool appEdit = false;
	bool export = false;
>;

float4x4 world : World
<
	bool appEdit = false;
	bool export = false;
>;




//-----------------------------------------------------------------------
//
// Textures
//
texture diffuseTexture : TEXTURE							// Diffuse colour in RGB, translucency in alpha
<
	string UIName = "Diffuse Texture";
	bool appEdit = true;
>;

// identical to diffuse texture, but present as PS3 gets confused with setting this texture as it was previously named in multiple samplers
texture diffuseTextureAniso : TEXTURE						// Diffuse colour in RGB, translucency in alpha
<
	string UIName = "Diffuse Texture Aniso";
	bool appEdit = true;
>;



//-----------------------------------------------------------------------
//
// Variables
//
float specularFactor
<
	string UIName = "Global Specular Factor";
	bool appEdit = true;
> = 2.5f;

float specularPower
<
	string UIName = "Global Specular Power";
	bool appEdit = true;
> = 30.0f;

float startCamFadeDistToVertex
<
	bool appEdit = true;
> = 0.0f;

float endCamFadeDistToVertex
<
	bool appEdit = true;
> = 0.0f;

float dynamicFactor
<
	bool appEdit = true;
> = 0.0f;

//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture = "diffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Clamp";
	string AddressV  = "Wrap";
>
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Clamp;
	AddressV  = Wrap;
#endif
};

sampler2D diffuseMapAniso : SAMPLER
<
	SET_SRGB_TEXTURE
	bool appEdit = false;
	string SamplerTexture = "diffuseTextureAniso";
	string MinFilter = "Anisotropic";
	string MagFilter = "Anisotropic";
	string MipFilter = "Linear";
	string AddressU = "Clamp";
	string AddressV = "Wrap";
	int MaxAnisotropy = 6;
	int MipMapLODBias = 0;
>
= sampler_state
{
	Texture = < diffuseTextureAniso >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _ANISOMINFILTER;
	MagFilter = _ANISOMAXFILTER;
	MipFilter = Linear;
	AddressU = Clamp;
	AddressV = Wrap;
#if defined(_PS3_)
	LODBias = 0;
	#else
	MipMapLODBias = 0;
#endif
	SET_MAX_ANISOTROPY( 6 )
#endif
}; 



//-----------------------------------------------------------------------
//
// Lighting
//
#if defined(_XBOX) && !defined(_TOOLS_COMPILATION_)
#include "D:\Render\Shaders\lighting_globals.fxh"
#else
#include "..\..\..\Render\Shaders\lighting_globals.fxh"
#endif
DECLARE_LIGHTING_PARAMS




//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// TODO: put scale into vertex shader const when supported render-side
struct VSINPUT
{
	float3 position		: POSITION;												
	float4 colour		: COLOR0;												
	float2 texCoord		: TEXCOORD0;
};

struct VSOUTPUT
{
	float4 position		: POSITION;												
	float4 colour		: TEXCOORD1;												
	float2 texCoord		: TEXCOORD0;
};

struct VSOUTPUT_CAMFADE
{
	float4 position		: POSITION;												
	float4 colour		: TEXCOORD2;												
	float2 texCoord		: TEXCOORD0;
	float3 worldPos		: TEXCOORD1;	
};

struct VSOUTPUT_LIT
{
	float4 position		: POSITION;
	float4 colour		: TEXCOORD2;
	float2 texCoord		: TEXCOORD0;
	float4 eye 			: TEXCOORD1;

	DECLARE_LIGHTING_INTERPOLATORS( 3 )
};



//-----------------------------------------------------------------------
//
// Vertex shader code
//

VSOUTPUT_LIT VShader( VSINPUT _input )
{
	VSOUTPUT_LIT _output;
	
	// get vertex position in clip space
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	// Calculate world-space vector to the eye
	_output.eye = float4( worldCameraPos - worldPos, 0);

	// apply ambient shading 
	_output.colour = _input.colour;
	float3 normal = float3(0.0f, 1.0f, 0.0f);
	float3 worldEyeVec = worldCameraPos - worldPos;
	DO_VS_LIGHTING_CALCULATIONS
//	DO_VS_AMBIENT_LIGHTING_CALCULATIONS_TRANSITION
	
	// pass thru
	_output.texCoord = _input.texCoord;
		
	return _output;
}



VSOUTPUT VShaderUnlit( VSINPUT _input )
{
	VSOUTPUT _output;
	
	// get vertex position in clip space
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );
	
	// pass thru
	_output.colour = _input.colour;
	_output.texCoord = _input.texCoord;
		
	return _output;
}



VSOUTPUT_CAMFADE VShaderUnlitCamFade( VSINPUT _input )
{
	VSOUTPUT_CAMFADE _output;
	
	// get vertex position in clip space
	float4 inPos = float4( _input.position, 1.0f );
	_output.position = mul( inPos, worldviewproj );
	
	// fade particles as they come near the camera
	// TODO: the first line is correct code, but seeing as we never have anything but the identity matrix as 'world' I'll leave it out to save
	// cycles. If needed later for something else we might want to make a new technique.
	//_output.worldPos = mul(inPos, world);
	_output.worldPos = inPos;
		
	// pass thru
	_output.colour = _input.colour;
	_output.texCoord = _input.texCoord;
		
	return _output;
}


	

//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

struct PSINPUT
{
	float4 colour		: TEXCOORD1;												
	float2 texCoord		: TEXCOORD0;
};

struct PSINPUT_LIT
{
	float4 colour		: TEXCOORD2;												
	float2 texCoord		: TEXCOORD0;
	float4 eye 			: TEXCOORD1;

	DECLARE_LIGHTING_INTERPOLATORS( 3 )
	DECLARE_SHADOW_PS_INPUTS
};

struct PSINPUT_CAMFADE
{
	float4 colour		: TEXCOORD2;												
	float2 texCoord		: TEXCOORD0;
	float3 worldPos		: TEXCOORD1;	
};

struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

PSOUTPUT fragmentShader( PSINPUT_LIT _input )
{
	PSOUTPUT _output;	

	PS_GENERATE_WORLDPOS( _input.eye.xyz )

	// Read texture
	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );

	// modulate with interpolated vertex colour 
	float4 colour = diffuseTexColour * _input.colour;
	
	// write this now as it will get clobbered in DO_PS_LIGHTING_CALCULATIONS
	_output.colour.a = colour.a;

	// Values required by the lighting macros, 
	float3 eye = normalize( _input.eye.xyz );
	float4 specularTexColour = float4( 1.0f, 1.0f, 1.0f, 0.0f );			// Set this to the specular highlight colour
	float3 normal = float3( 0.0f, 1.0f, 0.0f );
	float globalSpecularFactorValue = specularFactor;
	float minSpecPowerValue = specularPower;
	float maxSpecPowerValue = specularPower;

	DO_PS_LIGHTING_CALCULATIONS( colour, _input.eye.xyz )

	_output.colour.rgb = colour.rgb;

	return _output;
}

PSOUTPUT fragmentShaderAniso( PSINPUT_LIT _input )
{
	PSOUTPUT _output;	

	PS_GENERATE_WORLDPOS( _input.eye.xyz )

	// Read texture
	float4 diffuseTexColour = tex2D( diffuseMapAniso, _input.texCoord );

	// modulate with interpolated vertex colour 
	float4 colour = diffuseTexColour * _input.colour;
	
	// write this now as it will get clobbered in DO_PS_LIGHTING_CALCULATIONS
	_output.colour.a = colour.a;

	// Values required by the lighting macros, 
	float3 eye = normalize( _input.eye.xyz );
	float4 specularTexColour = float4( 1.0f, 1.0f, 1.0f, 0.0f );			// Set this to the specular highlight colour
	float3 normal = float3( 0.0f, 1.0f, 0.0f );
	float globalSpecularFactorValue = specularFactor;
	float minSpecPowerValue = specularPower;
	float maxSpecPowerValue = specularPower;

	DO_PS_LIGHTING_CALCULATIONS( colour, _input.eye.xyz )

	_output.colour.rgb = colour.rgb;

	return _output;
}

PSOUTPUT fragmentShaderAnisoUnlit( PSINPUT _input )
{
	PSOUTPUT _output;	

	// Read textures
	float4 diffuseTex = tex2D( diffuseMapAniso, _input.texCoord );
		
	// modulate with interpolated vertex colour 
	_output.colour = diffuseTex * _input.colour;

	return _output;
}

PSOUTPUT fragmentShaderAnisoUnlitCamFade( PSINPUT_CAMFADE _input )
{
	PSOUTPUT _output;	

	// Read textures
	float4 diffuseTex = tex2D( diffuseMapAniso, _input.texCoord );
	
	// perform cam fade calc
	float3 diff = float3(_input.worldPos - worldCameraPos);
	float dist = length(diff);
	float alphaFactor = 1.0f - saturate((startCamFadeDistToVertex - (dist - endCamFadeDistToVertex)) / startCamFadeDistToVertex);
	_input.colour.a *= alphaFactor;
		
	// modulate with interpolated vertex colour 
	_output.colour = diffuseTex * _input.colour;

	return _output;
}

PSOUTPUT fragmentShaderAnisoUnlitCamFadeDynamic( PSINPUT_CAMFADE _input )
{
	PSOUTPUT _output;	

	// Read textures
	float4 diffuseTex = tex2D( diffuseMapAniso, _input.texCoord );
	
	// perform cam fade calc
	float3 diff = float3(_input.worldPos - worldCameraPos);
	float dist = length(diff);
	float alphaFactor = 1.0f - saturate((startCamFadeDistToVertex - (dist - endCamFadeDistToVertex)) / startCamFadeDistToVertex);

	// Compute the absolute [m/s] difference of input speed against expected speed.
	// Negative means we go slower than expected, positive means faster.
	// We know that speed is scaled to <0, 1> range using max speed 128.0f - see DynamicDrivingLine.cpp
	const float maxSpeed = 128.0f;
	const float expandedDynamic = dynamicFactor * maxSpeed;
	const float expandedExpected = _input.colour.r * maxSpeed;
	const float speedDifference = expandedDynamic - expandedExpected;

	// Define color ranges - see comment below.
	// The units are [m/s] difference against expected speed.
	const float underRange = -1.5f;
	const float bitOverRange = 0.0f;
	const float overRange = 5.0f;

	// How long it takes to lerp between colour changes.
	// Must be smaller than difference between cosequent Ranges,
	// otherwise the colour lerp would not be continuous. 
	const float speedFadeUnder = 1.5f;
	const float speedFadeBitOver = 1.0f;
	const float speedFadeOver = 1.5f;

	// Let the hinting color fade out to colDistant with distance.
	// Fragment within (against reference point) distance stays hint colored,
	// fragment outside blends to colDistant.
	const float distanceFadeStart = 60.0f;
	const float distanceFadeLength = 20.0f;
	const float distanceFactor = saturate( ( dist - distanceFadeStart ) / distanceFadeLength );

	// The actual colours we use:
	//
	// Percentual difference:
	//
	//        underRange 
	//        |           bitOverRange
	//        |           |       overRange
	// -------|-----0-----|-------|----------
	//  Under    Neutral   BitOver    Over
	//  (blue)   (green)   (yellow)  (red)
	//
	//
	// 'Under' colour   - player is under expected speed and should accelerate.
	// 'Neutral' colour - current speed matches expectation.
	// 'BitOver' colour - player is going bit too fast and should be aware or slightly deccelerate.
	// 'Over' colour    - player is going too fast and should deccelerate.
	//
	// 'Distant' colour is the colour the driving line fades into in distance.
	//
	const float3 COLOUR_UNDER = float3( 0.0f, 1.0f, 0.0f );
	const float3 COLOUR_NEUTRAL = float3( 0.0f, 1.0f, 0.0f );
	const float3 COLOUR_BITOVER = float3( 2.0f, 1.5f, 0.0f );
	const float3 COLOUR_OVER = float3( 1.5f, 0.0f, 0.0f );
	const float3 COLOUR_DISTANT = float3( 0.0f, 1.0f, 0.0f );

	float3 speedColour;

	if ( speedDifference < underRange )
	{
		const float blendFactor = saturate( abs( speedDifference - underRange ) / speedFadeUnder );
		speedColour = lerp( COLOUR_NEUTRAL, COLOUR_UNDER, blendFactor );
	}
	else if ( speedDifference > overRange )
	{
		const float blendFactor = saturate( abs( speedDifference - overRange ) / speedFadeOver );
		speedColour = lerp( COLOUR_NEUTRAL, COLOUR_OVER, blendFactor );
	}
	else if ( speedDifference > bitOverRange )
	{
		const float blendFactor = saturate( abs( speedDifference - bitOverRange ) / speedFadeBitOver );
		speedColour = lerp( COLOUR_NEUTRAL, COLOUR_BITOVER, blendFactor );
	}
	else
	{
		speedColour = COLOUR_NEUTRAL;
	}


	// Now we have computed dynamic color, blend it with distanc colour.
	const float3 lineColour = lerp( speedColour, COLOUR_DISTANT, distanceFactor );

	// The output is combination of the colour we devised and the texture.
	// Opacity is controlled by the distance fade factor computed at the top.
	const float lineOpacity = _input.colour.a * alphaFactor;
	_output.colour = diffuseTex * float4( lineColour, lineOpacity );

	return _output;
}

PSOUTPUT fragmentShaderUnlit( PSINPUT _input )
{
	PSOUTPUT _output;	

	// Read textures
	float4 diffuseTex = tex2D( diffuseMap, _input.texCoord );
		
	// modulate with interpolated vertex colour 
	_output.colour = diffuseTex * _input.colour;

	return _output;
}



//-----------------------------------------------------------------------
//
// Technique(s)
//

technique GenericQuad
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "GenericQuad";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_DONT_RENDER";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
	string lowDetailBehaviour	= "ERMB_DONT_RENDER";
>
{
	pass Pass0
	{
#ifdef _PS3_
		VertexShader = compile sce_vp_rsx VShader();
		PixelShader = compile sce_fp_rsx fragmentShader();
#else
		VertexShader = compile vs_3_0 VShader();
		PixelShader = compile ps_3_0 fragmentShader();
#endif
	}
}

technique GenericQuadAniso
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "GenericQuadAniso";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_DONT_RENDER";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
	string lowDetailBehaviour	= "ERMB_DONT_RENDER";
>
{
	pass Pass0
	{
#ifdef _PS3_
		VertexShader = compile sce_vp_rsx VShader();
		PixelShader = compile sce_fp_rsx fragmentShaderAniso();
#else
		VertexShader = compile vs_3_0 VShader();
		PixelShader = compile ps_3_0 fragmentShaderAniso();
#endif
	}
}

technique GenericQuadAnisoUnlit
<
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "GenericQuadAnisoUnlit";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_DONT_RENDER";
	string shadowGenBehaviour = "ERMB_DONT_RENDER";
	string lowDetailBehaviour	= "ERMB_DONT_RENDER";
>
{
	pass Pass0
	{
#ifdef _PS3_
		VertexShader = compile sce_vp_rsx VShaderUnlit();
		PixelShader = compile sce_fp_rsx fragmentShaderAnisoUnlit();
#else
		VertexShader = compile vs_3_0 VShaderUnlit();
		PixelShader = compile ps_3_0 fragmentShaderAnisoUnlit();
#endif
	}
}

technique GenericQuadAnisoUnlitCamFade
<
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "GenericQuadAnisoUnlitCamFade";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_DONT_RENDER";
	string shadowGenBehaviour = "ERMB_DONT_RENDER";
	string lowDetailBehaviour	= "ERMB_DONT_RENDER";
>
{
	pass Pass0
	{
#ifdef _PS3_
		VertexShader = compile sce_vp_rsx VShaderUnlitCamFade();
		PixelShader = compile sce_fp_rsx fragmentShaderAnisoUnlitCamFade();
#else
		VertexShader = compile vs_3_0 VShaderUnlitCamFade();
		PixelShader = compile ps_3_0 fragmentShaderAnisoUnlitCamFade();
#endif
	}
}

technique GenericQuadAnisoUnlitCamFadeDynamic
<
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "GenericQuadAnisoUnlitCamFadeDynamic";
	int    normalDeferredID		= 1;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_DONT_RENDER";
	string shadowGenBehaviour = "ERMB_DONT_RENDER";
	string lowDetailBehaviour	= "ERMB_DONT_RENDER";
>
{
	pass Pass0
	{
#ifdef _PS3_
		VertexShader = compile sce_vp_rsx VShaderUnlitCamFade();
		PixelShader = compile sce_fp_rsx fragmentShaderAnisoUnlitCamFadeDynamic();
#else
		VertexShader = compile vs_3_0 VShaderUnlitCamFade();
		PixelShader = compile ps_3_0 fragmentShaderAnisoUnlitCamFadeDynamic();
#endif
	}
}

technique GenericQuadUnlit
<
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "GenericQuadUnlit";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_DONT_RENDER";
	string shadowGenBehaviour = "ERMB_DONT_RENDER";
	string lowDetailBehaviour	= "ERMB_DONT_RENDER";
>
{
	pass Pass0
	{
#ifdef _PS3_
		VertexShader = compile sce_vp_rsx VShaderUnlit();
		PixelShader = compile sce_fp_rsx fragmentShaderUnlit();
#else
		VertexShader = compile vs_3_0 VShaderUnlit();
		PixelShader = compile ps_3_0 fragmentShaderUnlit();
#endif
	}
}


